/*
						A Simple Binary Genetic Algorithm (Binary SGA)
 						for solving the Mastermind Game
						Created by Stratos Georgopoulos

  based on the algorithm presented in the book:
  Michalewicz, Z., "Genetic Algorithms + Data Structures = Evolution Programs",
  Springer-Verlag, 3rd edition, 1996.

			  --------------- SGA_Mastermind.C  Source File ---------------
*/
#include "sga_mastermind.h"	/* include the sga header file */

/* ----------------------------- Random Number Generators --------------------------------- */

/*              RandVal                                 	  */
/* returns a pseudorandom real number between the supplied    */
/* low and high arguments.                              	  */
/*                                                      	  */
double RandVal (double low, double high)
{
  return ((((double) rand())/RAND_MAX)*(high-low))+low;
}

/*              IRand0N                                 */
/* returns a pseudorandom value where 0<= value <=N-1   */
/*                                                      */
int IRand0N(int N)
{
	return rand()%N;
}

/*              IRand                                   		 */
/* returns a pseudorandom integer number between the supplied    */
/* low and high arguments: Low <= value <= High             	 */
/*                                                      		 */
int IRand(int Low, int High)
{
	return Low+IRand0N(High-Low+1);
}

/* ---------------------------- Vector manipulation functions ---------------------------- */
/* 				BELONGS_INTO															   */
/* searches if num belongs into vector vec (where n equals the number of elements of vec)  */
/* if it belongs it returns 1 else it returns 0											   */
/*																						   */
int belongs_into(int num, int * vec, int n)
{
	int found=FALSE;

	for(int i=0; i<n && found==FALSE; i++)
		if(num == vec[i])
			found = TRUE;

	if(found == TRUE)
		return 1;
	else
		return 0;
}

/* ---------------------------- Genetic Algorithm's Functions ----------------------------- */

/*              INITIALIZE                                              */
/* Creates the intitial population of genotypes (chromosomes) in random */
/*                                                                      */
void initialize(void)
{
	/* assign a random integer value to each gene of each genotype */
  	for(int i=0;i<=POPSIZE;i++)
    	for(int j=0;j<NGENES;j++)
      		Population[i].Gene[j] = IRand0N(NCOLORS);
}


/*              				EVALUATE                                      	*/
/* calculate the fitness of each member of the population for the vector game 	*/
/*																				*/
void evaluate(int *Guess_vec)
{
	Best=POPSIZE;

  	for(int mem=0; mem<POPSIZE; mem++)
	{
		Population[mem].Fitness = 0; /* initialize fitness */
		for(int i=0; i<NGENES; i++)
			if (Population[mem].Gene[i] == Guess_vec[i])
				Population[mem].Fitness += 2;
			else if (belongs_into(Population[mem].Gene[i], Guess_vec, NGENES))
				Population[mem].Fitness += 1;

		/*	Population[mem].Fitness += (Population[mem].Gene[i] == Guess_vec[i]) ? 1 : 0;*/

    	/* Keep track of the best member of the population  */
    	/* Note that the last member of the population holds*/
    	/* a copy of the best member.                       */
    	if(Population[mem].Fitness > Population[POPSIZE].Fitness)
			Best=mem;
    }
	/* Copy the Best member found to the POPSIZE (last) place */
	if(Best != POPSIZE)
	{
		Population[POPSIZE].Fitness = Population[Best].Fitness;
		for(int i=0;i<NGENES;i++)
			Population[POPSIZE].Gene[i] = Population[Best].Gene[i];
	}
}


/*              COPY_GENOTYPES                                   */
/* 		Coppies a genotype to another                            */
/*                                                               */
void copy_genotypes(struct genotype* oldgenotype, struct genotype* newgenotype)
{
	for(int i=0; i<NGENES; i++)
    	newgenotype->Gene[i] = oldgenotype->Gene[i];

  	newgenotype->Fitness 		= oldgenotype->Fitness;
  	newgenotype->RFitness 		= oldgenotype->RFitness;
  	newgenotype->CFitness 		= oldgenotype->CFitness;
  	newgenotype->Selection_Prob = oldgenotype->Selection_Prob;
  	newgenotype->Cumulative_Prob= oldgenotype->Cumulative_Prob;

  	newgenotype->Survivor	= oldgenotype->Survivor;
  	newgenotype->Mate		= oldgenotype->Mate;
  	newgenotype->Mutate		= oldgenotype->Mutate;
}


/*              COPY_POPULATION                                 */
/* 		Copies a population to another population               */
/*                                                              */
void copy_population(struct genotype old_pop[POPSIZE+1],struct genotype new_pop[POPSIZE+1])
{
	for(int mem=0; mem<=POPSIZE; mem++)
    	copy_genotypes(&old_pop[mem],&new_pop[mem]);
}


/*              		ROULETTE WHEEL SELECTION                        */
/* This is an implementation of STANDARD PROPORTIONAL SELECTION 		*/
/* (or ROULETTE WHEEL SELECTION) with elitism for MAXIMIZATION problems */
/*                                                              		*/
void RouletteWheelSelection(void)
{
  	int    member, spin_num; /* Some counter variables       */
  	double Total_Fitness;         /* The total population fitness */
  	double roulette=0;
  	/* a variable for temporary storing of the population */
  	struct genotype Buffered_Pop[POPSIZE+1];
  	int Found;                    /* A flag */

  	/* First, find the total fitness of the population    */
  	Total_Fitness=0;
  	for(member=0; member<POPSIZE; member++)
      	Total_Fitness += Population[member].Fitness;

  	/* Next, calculate the probability of a selection of each genotype      */
  	for(member=0; member<POPSIZE; member++)
    	Population[member].Selection_Prob = Population[member].Fitness/Total_Fitness;

	/* Now, calculate the cumulative probability of each genotype     */
  	Population[0].Cumulative_Prob=Population[0].Selection_Prob;

  	for(member=1; member<POPSIZE; member++)
    	Population[member].Cumulative_Prob = Population[member-1].Cumulative_Prob +
                                             Population[member].Selection_Prob;

  	/* Finally, select the survivors using the cumulative probability */
  	/* and create the new Population                                  */
  	for(spin_num=0; spin_num<POPSIZE; spin_num++)
	{
    	Found = FALSE;
    	roulette = frand(); /* generate a random real number in [0, 1] */

    	if(roulette < Population[0].Cumulative_Prob) 			/* Select the first member of the Population */
      		copy_genotypes(&Population[0],&Buffered_Pop[spin_num]);
        else if(roulette >Population[POPSIZE-1].Cumulative_Prob) /* select the best member of the population */
             copy_genotypes(&Population[POPSIZE],&Buffered_Pop[spin_num]);
        else
        	for(member=1; member<POPSIZE && Found==FALSE; member++)
           		if( (roulette > Population[member-1].Cumulative_Prob) &&
              	    (roulette <=Population[member].Cumulative_Prob) )
           			{
              			copy_genotypes(&Population[member],&Buffered_Pop[spin_num]);
              			Found = TRUE;
           			}
	}

  	/* copy the best individual */
  	copy_genotypes(&Population[POPSIZE],&Buffered_Pop[POPSIZE]);

  	/* Copy the Buffered_Pop to the original Population */
  	copy_population(Buffered_Pop,Population);

  	/* Population , now is the new population           */
}


/*              			CROSSOVER                                   */
/* This is an implementation of STANDARD SINGLE POINT CROSSOVER	        */
/* Many other crossover operators developed may give better results		*/
/* For simplicity only the single point crossover is implemented here   */
/*                                                                      */
void crossover(void)
{
  	int mem,i,            /* Counting variables   */
      	parent1,          /* Parent one           */
      	parent2,          /* Parent two           */
      	xover_point,      /* Crossover point      */
      	count=0,
      	lovers=0,         /* number of matting genotypes */
      	indiv=0;
  	struct genotype temp_parent;

  	for(mem=0; mem<=POPSIZE; mem++)
    	Population[mem].Mate=FALSE;

  	/* first find the individuals for the Crossover operation */
  	for (mem=0; mem<POPSIZE; mem++)
    	if(frand() < PXOVER){ /* frand returns a random number in the range [0,1] */
       		Population[mem].Mate = TRUE;
       		lovers++;
    	}

	/* We want an even number of "lovers"*/
  	if((lovers%2) != 0)
	{
    	do  	/* find an non "lover" */
    		indiv=rand()%POPSIZE;
    	while(Population[indiv].Mate==TRUE);
  		/* make it "lover"    */
  		Population[indiv].Mate=TRUE;
  		lovers++;
	}

  	/* second mate the "lovers" */
  	mem=0;
  	for(count=0; count<(lovers/2); count++)
	{
    	while(Population[mem].Mate==FALSE) mem++; /* find the first parent */
    	parent1=mem;
		mem++;
    	while(Population[mem].Mate==FALSE) mem++; /* find the second parent */
    	parent2=mem;
		mem++;

    	/* select the crossover point :1...NGENES-1 */
    	xover_point=(rand()%(NGENES-1))+1;

    	/* Perform the crossover */
    	/* copy parent1 to temp_parent */
    	copy_genotypes(&Population[parent1],&temp_parent);

    	for(i=xover_point; i<NGENES; i++)
        	Population[parent1].Gene[i]=Population[parent2].Gene[i];
    	for(i=xover_point;i<NGENES;i++)
        	Population[parent2].Gene[i]=temp_parent.Gene[i];
  	}
  	/* set Mate flag to False */
	for(mem=0;mem<=POPSIZE;mem++)
    	Population[mem].Mate=FALSE;
}


/*              MUTATION                                                */
/* This is an implementation of random mutation. A gene selected       	*/
/* for mutation is getting a random integer value      					*/
/*                                                                      */
void mutate(void)
{
 	for(int member=0; member<POPSIZE; member++) /* for every member 	*/
    	for(int i=0; i<NGENES; i++)				/* and for every gene   */
      		if(frand()<PMUTATION)
         		Population[member].Gene[i] = IRand0N(NCOLORS);
}


/*              REPORT                                          */
/* Report progress of the simulation. Data is dumped to a log   */
/* file in comma separeted value format which can be imported   */
/* and graphed using any commercial spreadsheet package.        */
/*                                                              */
void report(FILE* GAlogfile)
{
  	double best_val,      /* Best population fitness              	*/
           avg,           /* Average population fitness             */
           stddev,        /* Std. deviation of population fitness   */
           sum_square,    /* Sum of the squares for std. dev calc.  */
           square_sum,    /* Square of the sums for std. dev. calc. */
           sum;           /* The summed population fitness          */
	int i=0;	/* counter */

	sum=0.0;
  	sum_square=0.0;

  	/* Calculate the summed population fitness and the sum        */
  	/* of the squared individual fitnesses.                       */
  	for(i=0; i<POPSIZE; i++){
    	sum += (Population[i].Fitness);
    	sum_square += pow(Population[i].Fitness, 2);
	}

  	/* Calculate the average and standard deviations of the       */
  	/* population's fitness.                                      */
  	avg=sum/(double)POPSIZE;
  	square_sum=sum*sum/(double) POPSIZE;
  	stddev=sqrt((1.0/(double)(POPSIZE-1))*(sum_square-square_sum));
  	best_val=Population[POPSIZE].Fitness;

  	/* Print the generation, best, avg, and std. deviation to a   */
  	/* file in csv format.                                        */
  	fprintf(GAlogfile,"\n%10d  %15.4f  %15.4f  %15.4f", Generation, best_val, avg, stddev);
	/* Print the Best Genotype */
	fprintf(GAlogfile, "  (");
	for(i=0;i<NGENES;i++)
    	fprintf(GAlogfile,"%d",Population[POPSIZE].Gene[i]);
	fprintf(GAlogfile, ") ");
}


/*                     MAIN                                     */
/* This is the main function. It loops for the specified number */
/* of generations. Each generation involves selecting survivors,*/
/* performing crossover and mutation, and then evaluating the   */
/* resulting population.                                        */
/*                                                              */
int main(void)
{
	int i,j;

  	srand(time(0));

	FILE *galog; /* the GA log file */
  	if((galog=fopen("galog.txt","w"))==NULL){
		printf("Can't open galog.txt \n");
    	exit(1);
  	}
	fprintf(galog,"%10s  %15s  %15s  %15s  %20s\n", "Generation", "Best Value", "Average", "StdDev", "Best Genotype");

  	Generation=0;
  	initialize();
  	printf("Initial Population\n");
	for(i=0;i<POPSIZE;i++){
    	for(j=0;j<NGENES;j++)
        	printf("%d", Population[i].Gene[j]);
    	printf("\n");
  	}
  	printf("Press any key to start the evolution\n");
	getch();

  	evaluate(Guess_Vector);
  	while((Population[POPSIZE].Fitness<2*NGENES) && (Generation<MAXGENS))
	{
    	Generation++;
    	RouletteWheelSelection();
    	crossover();
    	mutate();
    	evaluate(Guess_Vector);
    	printf("Generation : %d \n",Generation);
    	report(galog);
  	}

  	fprintf(galog,"\n\nSimulation completed\n");
  	fprintf(galog,"\n   Best member :\n");

  	for(i=0;i<NGENES;i++)
    	fprintf(galog,"\n Gene(%d) = %d",i,Population[POPSIZE].Gene[i]);

  	fclose(galog);
  	return 0;
}

/* ------------------------- THE END ---------------------------        */



